<?php

/**
 * Manage setting and expiring cached data
 *
 * Select actions can be used to force cached
 * data to expire. Implemented so far:
 *  - save_post
 *
 * When used in its ArrayAccess API the cache will provide non persistent storage.
 */
class Tribe__Cache implements ArrayAccess {
	const SCHEDULED_EVENT_DELETE_TRANSIENT = 'tribe_schedule_transient_purge';
	const NO_EXPIRATION  = 0;
	const NON_PERSISTENT = - 1;

	/**
	 * @var array
	 */
	protected $non_persistent_keys = array();

	/**
	 * Bootstrap hook
	 *
	 * @since 4.11.0
	 */
	public function hook() {
		if ( ! wp_next_scheduled( self::SCHEDULED_EVENT_DELETE_TRANSIENT ) ) {
			wp_schedule_event( time(), 'twicedaily', self::SCHEDULED_EVENT_DELETE_TRANSIENT );
		}

		add_action( self::SCHEDULED_EVENT_DELETE_TRANSIENT, [ $this, 'delete_expired_transients' ] );
	}

	public static function setup() {
		wp_cache_add_non_persistent_groups( array( 'tribe-events-non-persistent' ) );
	}

	/**
	 * @param string       $id
	 * @param mixed        $value
	 * @param int          $expiration
	 * @param string|array $expiration_trigger
	 *
	 * @return bool
	 */
	public function set( $id, $value, $expiration = 0, $expiration_trigger = '' ) {
		$key = $this->get_id( $id, $expiration_trigger );

		/**
		 * Filters the expiration for cache objects to provide the ability
		 * to make non-persistent objects be treated as persistent.
		 *
		 * @param int          $expiration         Cache expiration time.
		 * @param string       $id                 Cache ID.
		 * @param mixed        $value              Cache value.
		 * @param string|array $expiration_trigger Action that triggers automatic expiration.
		 * @param string       $key                Unique cache key based on Cache ID and expiration trigger last run time.
		 *
		 * @since 4.8
		 */
		$expiration = apply_filters( 'tribe_cache_expiration', $expiration, $id, $value, $expiration_trigger, $key );

		if ( self::NON_PERSISTENT === $expiration ) {
			$group      = 'tribe-events-non-persistent';
			$expiration = 1;

			// Add so we know what group to use in the future.
			$this->non_persistent_keys[] = $key;
		} else {
			$group = 'tribe-events';
		}

		return wp_cache_set( $key, $value, $group, $expiration );
	}

	/**
	 * @param              $id
	 * @param              $value
	 * @param int          $expiration
	 * @param string|array $expiration_trigger
	 *
	 * @return bool
	 */
	public function set_transient( $id, $value, $expiration = 0, $expiration_trigger = '' ) {
		return set_transient( $this->get_id( $id, $expiration_trigger ), $value, $expiration );
	}

	/**
	 * Get cached data. Optionally set data if not previously set.
	 *
	 * Note: When a default value or callback is specified, this value gets set in the cache.
	 *
	 * @param string       $id                 The key for the cached value.
	 * @param string|array $expiration_trigger Optional. Hook to trigger cache invalidation.
	 * @param mixed        $default            Optional. A default value or callback that returns a default value.
	 * @param int          $expiration         Optional. When the default value expires, if it gets set.
	 * @param mixed        $args               Optional. Args passed to callback.
	 *
	 * @return mixed
	 */
	public function get( $id, $expiration_trigger = '', $default = false, $expiration = 0, $args = array() ) {
		$flipped = array_flip( $this->non_persistent_keys );
		$group   = isset( $flipped[ $id ] ) ?
			'tribe-events-non-persistent'
			: 'tribe-events';
		$value   = wp_cache_get( $this->get_id( $id, $expiration_trigger ), $group );

		// Value found.
		if ( false !== $value ) {
			return $value;
		}

		if ( is_callable( $default ) ) {
			// A callback has been specified.
			$value = call_user_func_array( $default, $args );
		} else {
			// Default is a value.
			$value = $default;
		}

		// No need to set a cache value to false since non-existent values return false.
		if ( false !== $value ) {
			$this->set( $id, $value, $expiration, $expiration_trigger );
		}

		return $value;
	}

	/**
	 * @param string $id
	 * @param string|array $expiration_trigger
	 *
	 * @return mixed
	 */
	public function get_transient( $id, $expiration_trigger = '' ) {
		return get_transient( $this->get_id( $id, $expiration_trigger ) );
	}

	/**
	 * @param string $id
	 * @param string|array $expiration_trigger
	 *
	 * @return bool
	 */
	public function delete( $id, $expiration_trigger = '' ) {
		return wp_cache_delete( $this->get_id( $id, $expiration_trigger ), 'tribe-events' );
	}

	/**
	 * @param string $id
	 * @param string|array $expiration_trigger
	 *
	 * @return bool
	 */
	public function delete_transient( $id, $expiration_trigger = '' ) {
		return delete_transient( $this->get_id( $id, $expiration_trigger ) );
	}

	/**
	 * Purge all expired tribe_ transients.
	 *
	 * This uses a modification of the the query from https://core.trac.wordpress.org/ticket/20316
	 *
	 * @since 4.11.0
	 */
	public function delete_expired_transients() {
		global $wpdb;

		$time = time();

		$sql = "
			DELETE
				a,
				b
			FROM
				{$wpdb->options} a
				INNER JOIN {$wpdb->options} b
					ON b.option_name = CONCAT( '_transient_timeout_tribe_', SUBSTRING( a.option_name, 12 ) )
					AND b.option_value < {$time}
			WHERE
				a.option_name LIKE '\_transient_tribe\_%'
				AND a.option_name NOT LIKE '\_transient\_timeout_tribe\_%'
		";
		$wpdb->query( $sql );
	}

	/**
	 * @param string $key
	 * @param string|array $expiration_trigger
	 *
	 * @return string
	 */
	public function get_id( $key, $expiration_trigger = '' ) {
		if ( is_array( $expiration_trigger ) ) {
			$triggers = $expiration_trigger;
		} else {
			$triggers = array_filter( explode( '|', $expiration_trigger ) );
		}

		$last = 0;
		foreach ( $triggers as $trigger ) {
			// Bail on empty trigger otherwise it creates a `tribe_last_` opt on the DB.
			if ( empty( $trigger ) ) {
				continue;
			}

			$occurrence = $this->get_last_occurrence( $trigger );

			if ( $occurrence > $last ) {
				$last = $occurrence;
			}
		}

		$last = empty( $last ) ? '' : $last;
		$id   = $key . $last;
		if ( strlen( $id ) > 80 ) {
			$id = 'tribe_' . md5( $id );
		}

		return $id;
	}

	/**
	 * Returns the time of an action last occurrence.
	 *
	 * @param string $action The action to return the time for.
	 *
	 * @since 4.9.14 Changed the return value type from `int` to `float`.
	 *
	 * @return float The time (microtime) an action last occurred, or the current microtime if it never occurred.
	 */
	public function get_last_occurrence( $action ) {
		static $cache_var_name = __METHOD__;

		$cache_last_actions = tribe_get_var( $cache_var_name, [] );

		if ( isset( $cache_last_actions[ $action ] ) ) {
			return $cache_last_actions[ $action ];
		}

		$last_action = (float) get_option( 'tribe_last_' . $action, null );

		if ( ! $last_action ) {
			$last_action = microtime( true );

			update_option( 'tribe_last_' . $action, $last_action );
		}

		$cache_last_actions[ $action ] = (float) $last_action;

		tribe_set_var( $cache_var_name, $cache_last_actions );

		return $cache_last_actions[ $action ];
	}

	/**
	 * Sets the time (microtime) for an action last occurrence.
	 *
	 * @since 4.9.14 Changed the type of the time stored from an `int` to a `float`.
	 *
	 * @param string $action The action to record the last occurrence of.
	 * @param int    $timestamp The timestamp to assign to the action last occurrence or the current time (microtime).
	 */
	public function set_last_occurrence( $action, $timestamp = 0 ) {
		if ( empty( $timestamp ) ) {
			$timestamp = microtime( true );
		}
		update_option( 'tribe_last_' . $action, (float) $timestamp );

		$this->delete_expired_transients();
	}

	/**
	 * Builds a key from an array of components and an optional prefix.
	 *
	 * @param mixed  $components Either a single component of the key or an array of key components.
	 * @param string $prefix
	 * @param bool   $sort Whether component arrays should be sorted or not to generate the key; defaults to `true`.
	 *
	 * @return string The resulting key.
	 */
	public function make_key( $components, $prefix = '', $sort = true ) {
		$key = '';
		$components = is_array( $components ) ? $components : array( $components );
		foreach ( $components as $component ) {
			if ( $sort && is_array( $component ) ) {
				$is_associative = count( array_filter( array_keys( $component ), 'is_numeric' ) ) < count( array_keys( $component ) );
				if ( $is_associative ) {
					ksort( $component );
				} else {
					sort( $component );
				}
			}
			$key .= maybe_serialize( $component );
		}

		return $this->get_id( $prefix . md5( $key ) );
	}

	/**
	 * Whether a offset exists
	 *
	 * @link  http://php.net/manual/en/arrayaccess.offsetexists.php
	 * @param mixed $offset <p>
	 *                      An offset to check for.
	 *                      </p>
	 * @return boolean true on success or false on failure.
	 *                      </p>
	 *                      <p>
	 *                      The return value will be casted to boolean if non-boolean was returned.
	 * @since 4.11.0
	 */
	public function offsetExists( $offset ) {
		$flipped = array_flip( $this->non_persistent_keys );

		return isset( $flipped[ $offset ] );
	}

	/**
	 * Offset to retrieve
	 *
	 * @link  http://php.net/manual/en/arrayaccess.offsetget.php
	 * @param mixed $offset <p>
	 *                      The offset to retrieve.
	 *                      </p>
	 * @return mixed Can return all value types.
	 * @since 4.11.0
	 */
	public function offsetGet( $offset ) {
		return $this->get( $offset );
	}

	/**
	 * Offset to set
	 *
	 * @link  http://php.net/manual/en/arrayaccess.offsetset.php
	 * @param mixed $offset <p>
	 *                      The offset to assign the value to.
	 *                      </p>
	 * @param mixed $value  <p>
	 *                      The value to set.
	 *                      </p>
	 * @return void
	 * @since 4.11.0
	 */
	public function offsetSet( $offset, $value ) {
		$this->set( $offset, $value, self::NON_PERSISTENT );
	}

	/**
	 * Offset to unset
	 *
	 * @link  http://php.net/manual/en/arrayaccess.offsetunset.php
	 * @param mixed $offset <p>
	 *                      The offset to unset.
	 *                      </p>
	 * @return void
	 * @since 4.11.0
	 */
	public function offsetUnset( $offset ) {
		$this->delete( $offset );
	}

	/**
	 * Warms up the caches for a collection of posts.
	 *
	 * @since 4.10.2
	 *
	 * @param array|int $post_ids               A post ID, or a collection of post IDs.
	 * @param bool      $update_post_meta_cache Whether to warm-up the post meta cache for the posts or not.
	 */
	public function warmup_post_caches( $post_ids, $update_post_meta_cache = false ) {
		if ( empty( $post_ids ) ) {
			return;
		}

		$post_ids = (array) $post_ids;

		global $wpdb;

		$already_cached_ids = [];
		foreach ( $post_ids as $post_id ) {
			if ( wp_cache_get( $post_id, 'posts' ) instanceof \WP_Post ) {
				$already_cached_ids[] = $post_id;
			}
		}

		$required = array_diff( $post_ids, $already_cached_ids );

		if ( empty( $required ) ) {
			return;
		}

		/** @var Tribe__Feature_Detection $feature_detection */
		$feature_detection = tribe('feature-detection');
		$limit = $feature_detection->mysql_limit_for_example( 'post_result' );

		/**
		 * Filters the LIMIT that should be used to warm-up post caches and postmeta caches (if the
		 * `$update_post_meta_cache` parameter is `true`).
		 *
		 * Lower this value on less powerful hosts. Return `0` to disable the warm-up completely, and `-1` to remove the
		 * limit (not recommended).
		 *
		 * @since 4.10.2
		 *
		 * @param int $limit The number of posts whose caches will be warmed up, per query.
		 */
		$limit = (int) apply_filters( 'tribe_cache_warmup_post_cache_limit', min( $limit, count( $post_ids ) ) );

		if ( 0 === $limit ) {
			// Warmup disabled.
			return;
		}

		$buffer = $post_ids;
		$page   = 0;

		do {
			$limit_clause = $limit < 0 ? sprintf( 'LIMIT %d,%d', $limit * $page, $limit ) : '';
			$page ++;
			$these_ids    = array_splice( $buffer, 0, $limit );
			$interval     = implode( ',', array_map( 'absint', $these_ids ) );
			$posts_query  = "SELECT * FROM {$wpdb->posts} WHERE ID IN ({$interval}) {$limit_clause}";
			$post_objects = $wpdb->get_results( $posts_query );
			if ( is_array( $post_objects ) && ! empty( $post_objects ) ) {
				foreach ( $post_objects as $post_object ) {
					$post = new \WP_Post( $post_object );
					wp_cache_set( $post_object->ID, $post, 'posts' );
				}

				if ( $update_post_meta_cache ) {
					update_meta_cache( 'post', $these_ids );
				}
			}
		} while (
			! empty( $post_objects )
			&& is_array( $post_objects )
			&& count( $post_objects ) < count( $post_ids )
		);

    }
}
