<?php

/*
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * This software consists of voluntary contributions made by many individuals
 * and is licensed under the MIT license. For more information, see
 * <http://www.doctrine-project.org>.
 */
namespace MailPoetVendor\Doctrine\ORM\Cache\Persister\Entity;

if (!defined('ABSPATH')) exit;


use MailPoetVendor\Doctrine\ORM\Persisters\Entity\EntityPersister;
use MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata;
use MailPoetVendor\Doctrine\ORM\EntityManagerInterface;
use MailPoetVendor\Doctrine\ORM\Cache\ConcurrentRegion;
use MailPoetVendor\Doctrine\ORM\Cache\EntityCacheKey;
/**
 * Specific read-write entity persister
 *
 * @author Fabio B. Silva <fabio.bat.silva@gmail.com>
 * @author Guilherme Blanco <guilhermeblanco@hotmail.com>
 * @since 2.5
 */
class ReadWriteCachedEntityPersister extends \MailPoetVendor\Doctrine\ORM\Cache\Persister\Entity\AbstractEntityPersister
{
    /**
     * @param \MailPoetVendor\Doctrine\ORM\Persisters\Entity\EntityPersister $persister The entity persister to cache.
     * @param \MailPoetVendor\Doctrine\ORM\Cache\ConcurrentRegion            $region    The entity cache region.
     * @param \MailPoetVendor\Doctrine\ORM\EntityManagerInterface            $em        The entity manager.
     * @param \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata             $class     The entity metadata.
     */
    public function __construct(\MailPoetVendor\Doctrine\ORM\Persisters\Entity\EntityPersister $persister, \MailPoetVendor\Doctrine\ORM\Cache\ConcurrentRegion $region, \MailPoetVendor\Doctrine\ORM\EntityManagerInterface $em, \MailPoetVendor\Doctrine\ORM\Mapping\ClassMetadata $class)
    {
        parent::__construct($persister, $region, $em, $class);
    }
    /**
     * {@inheritdoc}
     */
    public function afterTransactionComplete()
    {
        $isChanged = \true;
        if (isset($this->queuedCache['update'])) {
            foreach ($this->queuedCache['update'] as $item) {
                $this->region->evict($item['key']);
                $isChanged = \true;
            }
        }
        if (isset($this->queuedCache['delete'])) {
            foreach ($this->queuedCache['delete'] as $item) {
                $this->region->evict($item['key']);
                $isChanged = \true;
            }
        }
        if ($isChanged) {
            $this->timestampRegion->update($this->timestampKey);
        }
        $this->queuedCache = array();
    }
    /**
     * {@inheritdoc}
     */
    public function afterTransactionRolledBack()
    {
        if (isset($this->queuedCache['update'])) {
            foreach ($this->queuedCache['update'] as $item) {
                $this->region->evict($item['key']);
            }
        }
        if (isset($this->queuedCache['delete'])) {
            foreach ($this->queuedCache['delete'] as $item) {
                $this->region->evict($item['key']);
            }
        }
        $this->queuedCache = array();
    }
    /**
     * {@inheritdoc}
     */
    public function delete($entity)
    {
        $key = new \MailPoetVendor\Doctrine\ORM\Cache\EntityCacheKey($this->class->rootEntityName, $this->uow->getEntityIdentifier($entity));
        $lock = $this->region->lock($key);
        $deleted = $this->persister->delete($entity);
        if ($deleted) {
            $this->region->evict($key);
        }
        if ($lock === null) {
            return $deleted;
        }
        $this->queuedCache['delete'][] = array('lock' => $lock, 'key' => $key);
        return $deleted;
    }
    /**
     * {@inheritdoc}
     */
    public function update($entity)
    {
        $key = new \MailPoetVendor\Doctrine\ORM\Cache\EntityCacheKey($this->class->rootEntityName, $this->uow->getEntityIdentifier($entity));
        $lock = $this->region->lock($key);
        $this->persister->update($entity);
        if ($lock === null) {
            return;
        }
        $this->queuedCache['update'][] = array('lock' => $lock, 'key' => $key);
    }
}
